# Copyright 2022 Huawei Technologies Co., Ltd
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""
Testing AdjustBrightness op in DE
"""
import numpy as np
from numpy.testing import assert_allclose

import mindspore.dataset as ds
import mindspore.dataset.transforms.transforms
import mindspore.dataset.vision as vision
from mindspore import log as logger
from util import diff_mse

DATA_DIR = "../data/dataset/testImageNetData/train/"
MNIST_DATA_DIR = "../data/dataset/testMnistData"

DATA_DIR_2 = ["../data/dataset/test_tf_file_3_images/train-0000-of-0001.data"]
SCHEMA_DIR = "../data/dataset/test_tf_file_3_images/datasetSchema.json"


def generate_numpy_random_rgb(shape):
    """
    Only generate floating points that are fractions like n / 256, since they
    are RGB pixels. Some low-precision floating point types in this test can't
    handle arbitrary precision floating points well.
    """
    return np.random.randint(0, 256, shape) / 255.


def test_adjust_brightness_eager(plot=False):
    """
    Feature: AdjustBrightness op
    Description: Test AdjustBrightness in eager mode
    Expectation: Output is the same as expected output
    """
    # Eager 3-channel
    image_file = "../data/dataset/testImageNetData/train/class1/1_1.jpg"
    img = np.fromfile(image_file, dtype=np.uint8)
    logger.info("Image.type: {}, Image.shape: {}".format(type(img), img.shape))

    img = vision.Decode()(img)
    img_adjustbrightness = vision.AdjustBrightness(1)(img)
    if plot:
        visualize_image(img, img_adjustbrightness)
    logger.info("Image.type: {}, Image.shape: {}".format(type(img_adjustbrightness),
                                                         img_adjustbrightness.shape))
    mse = diff_mse(img_adjustbrightness, img)
    logger.info("MSE= {}".format(str(mse)))
    assert mse == 0


def test_adjust_brightness_invalid_brightness_factor_param():
    """
    Feature: AdjustBrightness op
    Description: Test improper parameters for AdjustBrightness implementation
    Expectation: Throw ValueError exception and TypeError exception
    """
    logger.info("Test AdjustBrightness implementation with invalid ignore parameter")
    try:
        data_set = ds.ImageFolderDataset(dataset_dir=DATA_DIR, shuffle=False)
        trans = mindspore.dataset.transforms.transforms.Compose([
            vision.Decode(True),
            vision.Resize((224, 224)),
            vision.AdjustBrightness(brightness_factor=-10.0),
            vision.ToTensor()
        ])
        data_set = data_set.map(operations=[trans], input_columns=["image"])
    except ValueError as error:
        logger.info("Got an exception in AdjustBrightness: {}".format(str(error)))
        assert "Input brightness_factor is not within the required interval of " in str(error)
    try:
        data_set = ds.ImageFolderDataset(dataset_dir=DATA_DIR, shuffle=False)
        trans = mindspore.dataset.transforms.transforms.Compose([
            vision.Decode(True),
            vision.Resize((224, 224)),
            vision.AdjustBrightness(brightness_factor=[1, 2]),
            vision.ToTensor()
        ])
        data_set = data_set.map(operations=[trans], input_columns=["image"])
    except TypeError as error:
        logger.info("Got an exception in AdjustBrightness: {}".format(str(error)))
        assert "is not of type [<class 'float'>, <class 'int'>], but got" in str(error)


def test_adjust_brightness_pipeline():
    """
    Feature: AdjustBrightness op
    Description: Test AdjustBrightness in pipeline mode
    Expectation: Output is the same as expected output
    """
    # First dataset
    transforms1 = [vision.Decode(True), vision.Resize([64, 64]), vision.ToTensor()]
    transforms1 = mindspore.dataset.transforms.transforms.Compose(
        transforms1)
    ds1 = ds.TFRecordDataset(DATA_DIR_2,
                             SCHEMA_DIR,
                             columns_list=["image"],
                             shuffle=False)
    ds1 = ds1.map(operations=transforms1, input_columns=["image"])

    # Second dataset
    transforms2 = [
        vision.Decode(True),
        vision.Resize([64, 64]),
        vision.AdjustBrightness(1.0),
        vision.ToTensor()
    ]
    transform2 = mindspore.dataset.transforms.transforms.Compose(
        transforms2)
    ds2 = ds.TFRecordDataset(DATA_DIR_2,
                             SCHEMA_DIR,
                             columns_list=["image"],
                             shuffle=False)
    ds2 = ds2.map(operations=transform2, input_columns=["image"])

    num_iter = 0
    for data1, data2 in zip(ds1.create_dict_iterator(num_epochs=1),
                            ds2.create_dict_iterator(num_epochs=1)):
        num_iter += 1
        ori_img = data1["image"].asnumpy()
        cvt_img = data2["image"].asnumpy()
        assert_allclose(ori_img.flatten(),
                        cvt_img.flatten(),
                        rtol=1e-5,
                        atol=0)
        mse = diff_mse(ori_img, cvt_img)
        logger.info("MSE= {}".format(str(mse)))
        assert mse == 0


if __name__ == "__main__":
    test_adjust_brightness_eager()
    test_adjust_brightness_invalid_brightness_factor_param()
    test_adjust_brightness_pipeline()
