/**
 * Copyright 2020-2022 Huawei Technologies Co., Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "common/common.h"
#include "minddata/dataset/include/dataset/datasets.h"
#include "minddata/dataset/core/global_context.h"

#include "ir/dtype/type_id.h"

using namespace mindspore::dataset;

class MindDataTestPipeline : public UT::DatasetOpTesting {
 protected:
};

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage with default schema
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic1) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic1.";

  // Create a RandomDataset
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("image", mindspore::DataType::kNumberTypeUInt8, {2}));
  ASSERT_OK(schema->add_column("label", mindspore::DataType::kNumberTypeUInt8, {1}));
  std::shared_ptr<Dataset> ds = RandomData(50, schema);
  EXPECT_NE(ds, nullptr);

  ds = ds->SetNumWorkers(4);
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  ds = ds->Repeat(4);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    auto image = row["image"];
    auto label = row["label"];
    MS_LOG(INFO) << "Tensor image shape: " << image.Shape();
    MS_LOG(INFO) << "Tensor label shape: " << label.Shape();

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 200);

  // Manually terminate the pipeline
  iter->Stop();
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset with pipeline mode
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasicWithPipeline) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasicWithPipeline.";

  // Create two RandomDataset
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("image", mindspore::DataType::kNumberTypeUInt8, {2}));
  ASSERT_OK(schema->add_column("label", mindspore::DataType::kNumberTypeUInt8, {1}));
  std::shared_ptr<Dataset> ds1 = RandomData(50, schema);
  std::shared_ptr<Dataset> ds2 = RandomData(50, schema);
  EXPECT_NE(ds1, nullptr);
  EXPECT_NE(ds2, nullptr);

  // Create two Repeat operation on ds
  int32_t repeat_num = 2;
  ds1 = ds1->Repeat(repeat_num);
  EXPECT_NE(ds1, nullptr);
  repeat_num = 2;
  ds2 = ds2->Repeat(repeat_num);
  EXPECT_NE(ds2, nullptr);

  // Create two Project operation on ds
  std::vector<std::string> column_project = {"image", "label"};
  ds1 = ds1->Project(column_project);
  EXPECT_NE(ds1, nullptr);
  ds2 = ds2->Project(column_project);
  EXPECT_NE(ds2, nullptr);

  // Create a Concat operation on the ds
  ds1 = ds1->Concat({ds2});
  EXPECT_NE(ds1, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds1->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    auto image = row["image"];
    auto label = row["label"];
    MS_LOG(INFO) << "Tensor image shape: " << image.Shape();
    MS_LOG(INFO) << "Tensor label shape: " << label.Shape();

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 200);

  // Manually terminate the pipeline
  iter->Stop();
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset Getters method
/// Expectation: Output is equal to the expected output
TEST_F(MindDataTestPipeline, TestRandomDatasetGetters) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetGetters.";

  // Create a RandomDataset
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("image", mindspore::DataType::kNumberTypeUInt8, {2}));
  ASSERT_OK(schema->add_column("label", mindspore::DataType::kNumberTypeUInt8, {1}));
  std::shared_ptr<Dataset> ds = RandomData(50, schema);
  EXPECT_NE(ds, nullptr);

  std::vector<std::string> column_names = {"image", "label"};
  EXPECT_EQ(ds->GetDatasetSize(), 50);
  EXPECT_EQ(ds->GetColumnNames(), column_names);
}

TEST_F(MindDataTestPipeline, TestRandomDatasetBasic2) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic2.";

  // Create a RandomDataset
  std::shared_ptr<Dataset> ds = RandomData(10);
  EXPECT_NE(ds, nullptr);

  ds = ds->SetNumWorkers(1);
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  ds = ds->Repeat(2);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    // If no schema specified, RandomData will generate random columns
    // So we don't check columns here
    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 20);

  // Manually terminate the pipeline
  iter->Stop();
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage with SCHEMA_FILE as an input to Schema
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic3) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic3.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testTFTestAllTypes/datasetSchema.json";
  std::shared_ptr<SchemaObj> schema = Schema(SCHEMA_FILE);
  std::shared_ptr<Dataset> ds = RandomData(0, schema);
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  ds = ds->Repeat(2);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  std::vector<int64_t> expect_num = {1};
  std::vector<int64_t> expect_1d = {2};
  std::vector<int64_t> expect_2d = {2, 2};
  std::vector<int64_t> expect_3d = {2, 2, 2};

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    auto col_sint16 = row["col_sint16"];
    auto col_sint32 = row["col_sint32"];
    auto col_sint64 = row["col_sint64"];
    auto col_float = row["col_float"];
    auto col_1d = row["col_1d"];
    auto col_2d = row["col_2d"];
    auto col_3d = row["col_3d"];
    auto col_binary = row["col_binary"];

    // Validate shape
    ASSERT_EQ(col_sint16.Shape(), expect_num);
    ASSERT_EQ(col_sint32.Shape(), expect_num);
    ASSERT_EQ(col_sint64.Shape(), expect_num);
    ASSERT_EQ(col_float.Shape(), expect_num);
    ASSERT_EQ(col_1d.Shape(), expect_1d);
    ASSERT_EQ(col_2d.Shape(), expect_2d);
    ASSERT_EQ(col_3d.Shape(), expect_3d);
    ASSERT_EQ(col_binary.Shape(), expect_num);

    // Validate Rank
    ASSERT_EQ(col_sint16.Shape().size(), 1);
    ASSERT_EQ(col_sint32.Shape().size(), 1);
    ASSERT_EQ(col_sint64.Shape().size(), 1);
    ASSERT_EQ(col_float.Shape().size(), 1);
    ASSERT_EQ(col_1d.Shape().size(), 1);
    ASSERT_EQ(col_2d.Shape().size(), 2);
    ASSERT_EQ(col_3d.Shape().size(), 3);
    ASSERT_EQ(col_binary.Shape().size(), 1);

    // Validate type
    ASSERT_EQ(col_sint16.DataType(), mindspore::DataType::kNumberTypeInt16);
    ASSERT_EQ(col_sint32.DataType(), mindspore::DataType::kNumberTypeInt32);
    ASSERT_EQ(col_sint64.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_float.DataType(), mindspore::DataType::kNumberTypeFloat32);
    ASSERT_EQ(col_1d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_2d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_3d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_binary.DataType(), mindspore::DataType::kNumberTypeUInt8);

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 984);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage using SCHEMA_FILE path as an input
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic4) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic4.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testTFTestAllTypes/datasetSchema.json";
  std::shared_ptr<Dataset> ds = RandomData(0, SCHEMA_FILE);
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  ds = ds->Repeat(2);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  std::vector<int64_t> expect_num = {1};
  std::vector<int64_t> expect_1d = {2};
  std::vector<int64_t> expect_2d = {2, 2};
  std::vector<int64_t> expect_3d = {2, 2, 2};

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    auto col_sint16 = row["col_sint16"];
    auto col_sint32 = row["col_sint32"];
    auto col_sint64 = row["col_sint64"];
    auto col_float = row["col_float"];
    auto col_1d = row["col_1d"];
    auto col_2d = row["col_2d"];
    auto col_3d = row["col_3d"];
    auto col_binary = row["col_binary"];

    // Validate shape
    ASSERT_EQ(col_sint16.Shape(), expect_num);
    ASSERT_EQ(col_sint32.Shape(), expect_num);
    ASSERT_EQ(col_sint64.Shape(), expect_num);
    ASSERT_EQ(col_float.Shape(), expect_num);
    ASSERT_EQ(col_1d.Shape(), expect_1d);
    ASSERT_EQ(col_2d.Shape(), expect_2d);
    ASSERT_EQ(col_3d.Shape(), expect_3d);
    ASSERT_EQ(col_binary.Shape(), expect_num);

    // Validate Rank
    ASSERT_EQ(col_sint16.Shape().size(), 1);
    ASSERT_EQ(col_sint32.Shape().size(), 1);
    ASSERT_EQ(col_sint64.Shape().size(), 1);
    ASSERT_EQ(col_float.Shape().size(), 1);
    ASSERT_EQ(col_1d.Shape().size(), 1);
    ASSERT_EQ(col_2d.Shape().size(), 2);
    ASSERT_EQ(col_3d.Shape().size(), 3);
    ASSERT_EQ(col_binary.Shape().size(), 1);

    // Validate type
    ASSERT_EQ(col_sint16.DataType(), mindspore::DataType::kNumberTypeInt16);
    ASSERT_EQ(col_sint32.DataType(), mindspore::DataType::kNumberTypeInt32);
    ASSERT_EQ(col_sint64.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_float.DataType(), mindspore::DataType::kNumberTypeFloat32);
    ASSERT_EQ(col_1d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_2d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_3d.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_binary.DataType(), mindspore::DataType::kNumberTypeUInt8);

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 984);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage with input columns
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic5) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic5.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testTFTestAllTypes/datasetSchema.json";
  std::shared_ptr<Dataset> ds = RandomData(0, SCHEMA_FILE, {"col_sint32", "col_sint64", "col_1d"});
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  ds = ds->Repeat(2);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  std::vector<int64_t> expect_num = {1};
  std::vector<int64_t> expect_1d = {2};

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    EXPECT_EQ(row.size(), 3);

    auto col_sint32 = row["col_sint32"];
    auto col_sint64 = row["col_sint64"];
    auto col_1d = row["col_1d"];

    // Validate shape
    ASSERT_EQ(col_sint32.Shape(), expect_num);
    ASSERT_EQ(col_sint64.Shape(), expect_num);
    ASSERT_EQ(col_1d.Shape(), expect_1d);

    // Validate Rank
    ASSERT_EQ(col_sint32.Shape().size(), 1);
    ASSERT_EQ(col_sint64.Shape().size(), 1);
    ASSERT_EQ(col_1d.Shape().size(), 1);

    // Validate type
    ASSERT_EQ(col_sint32.DataType(), mindspore::DataType::kNumberTypeInt32);
    ASSERT_EQ(col_sint64.DataType(), mindspore::DataType::kNumberTypeInt64);
    ASSERT_EQ(col_1d.DataType(), mindspore::DataType::kNumberTypeInt64);

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 984);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage with null schema and input columns
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic6) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic6.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testTFTestAllTypes/datasetSchema.json";
  std::shared_ptr<Dataset> ds = RandomData(10, nullptr, {"col_sint32", "col_sint64", "col_1d"});
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 10);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset basic usage with empty string schema and input columns
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic7) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic7.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testTFTestAllTypes/datasetSchema.json";
  std::shared_ptr<Dataset> ds = RandomData(10, "", {"col_sint32", "col_sint64", "col_1d"});
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 10);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

// Feature: Test Repeat and Shuffle on RandomData
// Description: Apply operations, iterate through dataset and count rows
// Expectation: There should be 30 rows in the dataset
TEST_F(MindDataTestPipeline, TestRandomDatasetBasic8) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetBasic8.";

  // Create a RandomDataset
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(246);

  std::string SCHEMA_FILE = datasets_root_path_ + "/testRandomData/datasetSchema2.json";
  std::shared_ptr<Dataset> ds = RandomData(10, SCHEMA_FILE);
  EXPECT_NE(ds, nullptr);

  // Create a Shuffle operation on ds
  int32_t shuffle_size = 4;
  ds = ds->Shuffle(shuffle_size);
  EXPECT_NE(ds, nullptr);

  // Create a Repeat operation on ds
  int32_t repeat_num = 3;
  ds = ds->Repeat(repeat_num);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  // Check if RandomData() read correct columns
  uint64_t i = 0;
  while (row.size() != 0) {
    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 30);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset with UInt8 numbers for given shape
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetUInt8) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetUInt8.";

  // Create a RandomDataset with UInt8 numbers for given shape
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(963);
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("col1", mindspore::DataType::kNumberTypeUInt8, {4}));
  std::shared_ptr<Dataset> ds = RandomData(3, schema);
  EXPECT_NE(ds, nullptr);
  ds = ds->SetNumWorkers(3);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  uint64_t i = 0;
  while (row.size() != 0) {
    auto ind = row["col1"];
    TEST_MS_LOG_MSTENSOR(INFO, "ind: ", ind);

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 3);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset with float numbers for given shape
/// Expectation: The data is processed successfully
TEST_F(MindDataTestPipeline, TestRandomDatasetFloat) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetFloat.";

  // Create a RandomDataset with Float numbers for given shape
  u_int32_t curr_seed = GlobalContext::config_manager()->seed();
  GlobalContext::config_manager()->set_seed(369);
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("col1", mindspore::DataType::kNumberTypeFloat16, {2, 3}));
  std::shared_ptr<Dataset> ds = RandomData(4, schema);
  EXPECT_NE(ds, nullptr);
  ds = ds->SetNumWorkers(2);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  ASSERT_OK(iter->GetNextRow(&row));

  uint64_t i = 0;
  while (row.size() != 0) {
    auto ind = row["col1"];
    TEST_MS_LOG_MSTENSOR(INFO, "ind: ", ind);

    ASSERT_OK(iter->GetNextRow(&row));
    i++;
  }

  EXPECT_EQ(i, 4);

  // Manually terminate the pipeline
  iter->Stop();
  GlobalContext::config_manager()->set_seed(curr_seed);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset with duplicate column names
/// Expectation: Error message is logged, and CreateIterator() for invalid pipeline returns nullptr
TEST_F(MindDataTestPipeline, TestRandomDatasetDuplicateColumnName) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetDuplicateColumnName.";

  // Create a RandomDataset
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(schema->add_column("image", mindspore::DataType::kNumberTypeUInt8, {2}));
  ASSERT_OK(schema->add_column("label", mindspore::DataType::kNumberTypeUInt8, {1}));
  std::shared_ptr<Dataset> ds = RandomData(50, schema, {"image", "image"});
  // Expect failure: duplicate column names
  EXPECT_EQ(ds->CreateIterator(), nullptr);
}

/// Feature: RandomDataDataset
/// Description: Test RandomDataDataset with num_workers greater than num_rows
/// Expectation: Error message is logged, and CreateIterator() for invalid pipeline returns nullptr
TEST_F(MindDataTestPipeline, TestRandomDatasetFail) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDatasetFail.";
  // this will fail because num_workers is greater than num_rows
  std::shared_ptr<Dataset> ds = RandomData(3)->SetNumWorkers(5);
  EXPECT_EQ(ds->CreateIterator(), nullptr);
}

/// Feature: RandomData
/// Description: Test RandomData failed with large tensor shape
/// Expectation: Expecting error message is logged
TEST_F(MindDataTestPipeline, TestRandomDataLargeShape) {
  MS_LOG(INFO) << "Doing MindDataTestPipeline-TestRandomDataLargeShape.";

  // Create a RandomDataset
  std::shared_ptr<SchemaObj> schema = Schema();
  ASSERT_OK(
    schema->add_column("data", mindspore::DataType::kNumberTypeFloat32, {1, 12, 5, 3, 4, 5, 6, 7, 8, 5, 3, 1, 5, 6}));
  std::shared_ptr<Dataset> ds = RandomData(50, schema);
  EXPECT_NE(ds, nullptr);

  // Create an iterator over the result of the above dataset
  // This will trigger the creation of the Execution Tree and launch it.
  std::shared_ptr<Iterator> iter = ds->CreateIterator();
  EXPECT_NE(iter, nullptr);

  // Iterate the dataset and get each row
  std::unordered_map<std::string, mindspore::MSTensor> row;
  Status s = iter->GetNextRow(&row);
  ASSERT_ERROR(s);
  ASSERT_TRUE(s.ToString().find("memory size of total data can not be zero or exceed") != std::string::npos);
  ASSERT_TRUE(s.StatusCode() == StatusCode::kMDUnexpectedError);

  // Manually terminate the pipeline
  iter->Stop();
}
