# Copyright 2020 Huawei Technologies Co., Ltd
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ============================================================================
import numpy as np
from tests.st.control.cases_register import case_register
from mindspore.common import dtype as mstype
from mindspore import jit
from mindspore import nn
from mindspore import Tensor
from mindspore.ops import composite as C
from mindspore import context
from mindspore.common.parameter import Parameter
from mindspore.ops import functional as F

context.set_context(mode=context.GRAPH_MODE)


class ForwardNet(nn.Cell):
    def __init__(self, max_cycles=10):
        super(ForwardNet, self).__init__()
        self.max_cycles = max_cycles
        self.zero = Tensor(np.array(0), mstype.int32)
        self.weight = Parameter(Tensor(np.array(0), mstype.int32))

    def construct(self, x, y):
        out = self.zero
        for i in range(self.max_cycles):
            if out <= 20:
                self.weight = out
                F.assign(self.weight, i)
                out = x * y + out
        return out, self.weight


class BackwardNet(nn.Cell):
    def __init__(self, net):
        super(BackwardNet, self).__init__(auto_prefix=False)
        self.forward_net = net
        self.grad = C.GradOperation(get_all=True)

    def construct(self, *inputs):
        grads = self.grad(self.forward_net)(*inputs)
        return grads


@case_register.level1
@case_register.target_gpu
@case_register.target_ascend
def test_forward():
    x = Tensor(np.array(1), mstype.int32)
    y = Tensor(np.array(3), mstype.int32)
    # Graph Mode
    context.set_context(mode=context.GRAPH_MODE)
    graph_forward_net = ForwardNet(max_cycles=3)
    graph_mode_out = graph_forward_net(x, y)

    expect = (Tensor(np.array(9), mstype.int32), Tensor(np.array(2), mstype.int32))
    assert graph_mode_out == expect


@case_register.level1
@case_register.target_gpu
@case_register.target_ascend
def test_backward():
    x = Tensor(np.array(1), mstype.int32)
    y = Tensor(np.array(3), mstype.int32)
    # Graph Mode
    context.set_context(mode=context.GRAPH_MODE)
    graph_forward_net = ForwardNet(max_cycles=3)
    graph_backward_net = BackwardNet(graph_forward_net)
    graph_mode_grads = graph_backward_net(x, y)

    expect = (Tensor(np.array(9), mstype.int32), Tensor(np.array(3), mstype.int32))
    assert graph_mode_grads == expect


def test_if_in_for_dict():
    """
    Feature: Dictionary in for of control flow
    Description: Execute 'for x in xs' when xs is dictionary.
    Expectation: No exception.
    """
    @jit
    def control_flow_for(xs):
        result = 0
        ys = {'b': 0, 'g': 0}
        for x in xs:
            if x >= 'b':
                result += 1
        for y in ys:
            if y == 'b':
                result -= 1
        return result

    x = {'a': 1, 'd': 100, 'b': -10, 'c': 0}
    res = control_flow_for(x)
    assert res == 2
