/**
 * Copyright 2021 Huawei Technologies Co., Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "async/uuid_base.h"
#include <memory.h>
#include <atomic>
#include <random>

namespace mindspore {
namespace uuids {
constexpr int DASH_POS0 = 4;
constexpr int DASH_POS1 = 6;
constexpr int DASH_POS2 = 8;
constexpr int DASH_POS3 = 10;
constexpr int SHIFT_BIT = 4;

const uint8_t *uuid::BeginAddress() const { return uuidData; }

const uint8_t *uuid::EndAddress() const { return uuidData + UUID_SIZE; }

std::size_t uuid::Size() { return UUID_SIZE; }

std::string uuid::ToBytes(const uuid &u) {
  MINDRT_ASSERT(sizeof(u) == UUID_SIZE);
  return std::string(reinterpret_cast<const char *>(u.uuidData), sizeof(u.uuidData));
}

Option<unsigned char> uuid::GetValue(char c) {
  static char const digitsBegin[] = "0123456789abcdefABCDEF";
  static const size_t digitsLen = (sizeof(digitsBegin) / sizeof(char)) - 1;
  static const char *const digitsEnd = digitsBegin + digitsLen;
  static unsigned char const values[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 10, 11, 12, 13, 14, 15};
  size_t pos = std::find(digitsBegin, digitsEnd, c) - digitsBegin;
  if (pos >= digitsLen) {
    MS_LOG(ERROR) << "invalid char";
    return Option<unsigned char>(MindrtNone());
  }
  return Option<unsigned char>(values[pos]);
}

Option<uuid> uuid::FromString(const std::string &s) {
  auto sBegin = s.begin();
  if (sBegin == s.end()) {
    return Option<uuid>(MindrtNone());
  }
  auto c = *sBegin;
  bool hasOpenBrace = (c == '{');
  bool hasDashes = false;
  if (hasOpenBrace) {
    ++sBegin;
  }
  uuid u;
  for (size_t i = 0; sBegin != s.end(); ++i) {
    c = *(sBegin++);
    if ((i == DASH_POS0) && (c == '-')) {
      hasDashes = true;
      c = *(sBegin++);
    } else if ((i == DASH_POS1 || i == DASH_POS2 || i == DASH_POS3) && (hasDashes == true)) {
      if (c == '-' && sBegin != s.end()) {
        c = *(sBegin++);
      } else {
        MS_LOG(ERROR) << "str invalid";
        return Option<uuid>(MindrtNone());
      }
    }
    Option<unsigned char> oc1 = GetValue(c);
    if (oc1.IsNone()) {
      return Option<uuid>(MindrtNone());
    }
    u.uuidData[i] = oc1.Get();
    if (sBegin != s.end()) {
      c = *(sBegin++);
    }
    u.uuidData[i] <<= SHIFT_BIT;
    Option<unsigned char> oc2 = GetValue(c);
    if (oc2.IsNone()) {
      return Option<uuid>(MindrtNone());
    }
    u.uuidData[i] |= oc2.Get();
  }
  if ((hasOpenBrace && (c != '}')) || (sBegin != s.end())) {
    MS_LOG(ERROR) << "No } end or leng invalid";
    return Option<uuid>(MindrtNone());
  }
  return Option<uuid>(u);
}

// To check whether uuid looks like 0000000-000-000-000-000000000000000
bool uuid::IsNilUUID() const {
  for (std::size_t i = 0; i < Size(); i++) {
    if (uuidData[i]) {
      return false;
    }
  }
  return true;
}

const uint8_t *uuid::Get() const { return uuidData; }

uint8_t *uuid::BeginAddress() { return uuidData; }

uint8_t *uuid::EndAddress() { return uuidData + UUID_SIZE; }
}  // namespace uuids
}  // namespace mindspore
