/**
 * This is the C++ adaptation and derivative work of Myia (https://github.com/mila-iqia/myia/).
 *
 * Copyright 2019 Huawei Technologies Co., Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "abstract/dshape.h"
#include "utils/shape_utils.h"

namespace mindspore {
namespace abstract {
namespace {
std::string ShapeVectorToStr(const std::vector<int64_t> &shp) {
  std::ostringstream buffer;
  bool f_begin = true;
  buffer << "(";
  for (auto &x : shp) {
    if (!f_begin) {
      buffer << ", ";
    } else {
      f_begin = false;
    }
    buffer << x;
  }
  buffer << ")";
  return buffer.str();
}
}  // namespace

// used for print BaseShape content
std::ostream &operator<<(std::ostream &os, const BaseShape &bs) {
  os << bs.ToString();
  return os;
}

std::ostream &operator<<(std::ostream &os, const std::shared_ptr<BaseShape> bs) {
  MS_EXCEPTION_IF_NULL(bs);
  os << bs->ToString();
  return os;
}

bool BaseShape::operator==(const BaseShape &other) const { return tid() == other.tid(); }

bool BaseShape::operator!=(const BaseShape &other) const { return !(*this == other); }

std::string Shape::ToString() const {
  std::ostringstream buffer;
  buffer << ShapeVectorToStr(shape_);
  return buffer.str();
}

std::string Shape::DumpText() const {
  std::ostringstream buffer;
  buffer << "[";
  for (size_t i = 0; i < shape_.size(); i++) {
    buffer << (i > 0 ? ", " : "") << shape_[i];
  }
  buffer << "]";
  return buffer.str();
}

bool Shape::IsDynamic() const { return mindspore::IsDynamic(shape_); }

bool Shape::operator==(const BaseShape &other) const {
  if (tid() != other.tid()) {
    return false;
  }
  Shape other_shape = static_cast<const Shape &>(other);
  if (shape_ != other_shape.shape_) {
    return false;
  }
  return true;
}

void Shape::Broaden() {
  for (size_t i = 0; i < shape_.size(); i++) {
    shape_[i] = kShapeDimAny;
  }
}

bool DynamicSequenceShape::IsDynamic() const {
  if (element_shape_ == nullptr) {
    return false;
  }
  return element_shape_->IsDynamic();
}

bool DynamicSequenceShape::IsDimZero() const {
  if (element_shape_ == nullptr) {
    return false;
  }
  return element_shape_->IsDimZero();
}

bool DynamicSequenceShape::IsDimUnknown() const {
  if (element_shape_ == nullptr) {
    return false;
  }
  return element_shape_->IsDimUnknown();
}

size_t DynamicSequenceShape::hash() const {
  auto hash_code = static_cast<std::size_t>(tid());
  hash_code = hash_combine(hash_code, element_shape_->hash());
  return hash_code;
}

bool DynamicSequenceShape::operator==(const BaseShape &other) const {
  if (!other.isa<DynamicSequenceShape>()) {
    return false;
  }
  const auto &other_shape = dynamic_cast<const DynamicSequenceShape &>(other);
  return element_shape_ == other_shape.element_shape_;
}

std::string SequenceShape::ToString() const {
  std::ostringstream buffer;
  bool f_begin = true;
  for (const auto &p_shp : p_shapes_) {
    if (!f_begin) {
      buffer << ", ";
    } else {
      f_begin = false;
    }
    MS_EXCEPTION_IF_NULL(p_shp);
    buffer << p_shp->ToString();
  }
  return buffer.str();
}

BaseShapePtrList SequenceShape::ElementsClone() const {
  BaseShapePtrList ele_list;
  for (const auto &p_shp : p_shapes_) {
    MS_EXCEPTION_IF_NULL(p_shp);
    ele_list.push_back(p_shp->Clone());
  }
  return ele_list;
}

template bool SequenceShape::SequenceEqual<TupleShape>(const BaseShape &) const;
template bool SequenceShape::SequenceEqual<ListShape>(const BaseShape &) const;
}  // namespace abstract
}  // namespace mindspore
